import { useEffect, useState } from "react";
import { useDispatch, useSelector } from "react-redux";
import { useNavigate } from "react-router-dom";
import { getTutorData, tutorRegister } from "../../Redux/tutor/action";

// Component imports
import Navbar from "../../Components/Sidebar/Navbar";
import Header from "../../Components/Header/Header";
import AddIcon from "../../Components/AddIcon/AddIcon";
import TutorRow from "../../Components/Table/TutorRow";

// CSS & UI imports
import { Button, Drawer, Space, Spin, message } from "antd";
import "./Tutor.css";

const Tutor = () => {
  const navigate = useNavigate();
  const dispatch = useDispatch();

  // Redux states
  const {
    data: { isAuthenticated },
  } = useSelector((store) => store.auth);
  const { tutors, load } = useSelector((store) => store.tutor);

  // Local states
  const [filterTutor, setFilterTutor] = useState("");
  const [loading, setLoading] = useState(false);
  const [open, setOpen] = useState(false);
  const [messageApi, contextHolder] = message.useMessage();

  // Form data
  const initialFormData = {
    name: "",
    email: "",
    password: "",
    subject: "",
  };
  const [formData, setFormData] = useState(initialFormData);

  // Drawer toggle
  const showDrawer = () => setOpen(true);
  const onClose = () => {
    setOpen(false);
    dispatch(getTutorData(filterTutor)); // ✅ Auto refresh after closing drawer
  };

  // Input change handler
  const handleInputChange = (e) => {
    setFormData({ ...formData, [e.target.name]: e.target.value });
  };

  // Submit handler
  const handleSubmit = (e) => {
    e.preventDefault();

    if (formData.subject === "") {
      return messageApi.open({
        type: "info",
        content: "Please select a subject",
        duration: 3,
      });
    }

    setLoading(true);

    dispatch(tutorRegister(formData))
      .then((res) => {
        if (res?.msg === "User already registered") {
          messageApi.open({
            type: "info",
            content: "User already registered",
            duration: 3,
          });
        } else if (res?.msg === "Tutor Registration failed") {
          messageApi.open({
            type: "error",
            content: "Tutor Registration failed",
            duration: 3,
          });
        } else {
          messageApi.open({
            type: "success",
            content: "Tutor registered successfully!",
            duration: 3,
          });
          messageApi.open({
            type: "success",
            content: "Password sent over mail.",
            duration: 3,
          });

          // ✅ Reset form
          setFormData(initialFormData);
          onClose();

          // ✅ Refresh tutor list after success
          dispatch(getTutorData(filterTutor));
        }
      })
      .catch((error) => {
        console.error("Tutor Registration Error:", error);
        messageApi.open({
          type: "error",
          content: "Something went wrong during registration.",
          duration: 3,
        });
      })
      .finally(() => setLoading(false));
  };

  // Fetch tutor data when filter changes
  useEffect(() => {
    dispatch(getTutorData(filterTutor));
  }, [filterTutor, dispatch]);

  // Redirect if not authenticated
  useEffect(() => {
    if (!isAuthenticated) {
      navigate("/");
    }
  }, [isAuthenticated, navigate]);

  return (
    <Navbar>
      {contextHolder}
      <div className="admin">
        <Header Title="Tutor Data" Address="Tutor" />

        {/* Filter Dropdown */}
        <select
          style={{
            width: "200px",
            display: "flex",
            justifyContent: "center",
            alignItems: "center",
            margin: "20px auto 10px auto",
          }}
          value={filterTutor}
          onChange={(e) => setFilterTutor(e.target.value)}
        >
          <option value="">Filter by Subject</option>
          <option value="Maths">Maths</option>
          <option value="Physics">Physics</option>
          <option value="Chemistry">Chemistry</option>
          <option value="Biology">Biology</option>
          <option value="Political science">Political science</option>
          <option value="History">History</option>
        </select>

        {/* Tutor Table */}
        <div className="adminData">
          <section className="tableBody">
            <table>
              <thead>
                <tr>
                  <th>Name</th>
                  <th>Email</th>
                  <th>Subject</th>
                  <th>Access</th>
                  <th>Edit</th>
                  <th>Delete</th>
                </tr>
              </thead>
              <tbody>
                {tutors?.length > 0 ? (
                  tutors.map((data, i) => <TutorRow data={data} key={i} />)
                ) : (
                  <tr>
                    <td colSpan="6" style={{ textAlign: "center" }}>
                      No tutors available
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </section>
        </div>

        {/* Add Tutor Button */}
        <div onClick={showDrawer}>
          <AddIcon />
        </div>

        {/* Drawer to add new tutor */}
        <Drawer
          title="Create a new account"
          width={720}
          onClose={onClose}
          open={open}
          style={{ paddingBottom: 80 }}
          extra={
            <Space>
              <Button onClick={onClose}>Cancel</Button>
            </Space>
          }
        >
          <form onSubmit={handleSubmit}>
            <input
              required
              name="name"
              type="text"
              value={formData.name}
              placeholder="Enter Name"
              onChange={handleInputChange}
            />
            <input
              required
              name="email"
              type="email"
              value={formData.email}
              placeholder="Enter Email"
              onChange={handleInputChange}
            />
            <input
              required
              name="password"
              type="password"
              value={formData.password}
              placeholder="Enter Password"
              onChange={handleInputChange}
            />
            <select
              required
              name="subject"
              value={formData.subject}
              onChange={handleInputChange}
            >
              <option value="">Choose Subject</option>
              <option value="Maths">Maths</option>
              <option value="Physics">Physics</option>
              <option value="Chemistry">Chemistry</option>
              <option value="Biology">Biology</option>
              <option value="Political science">Political science</option>
              <option value="History">History</option>
            </select>
            <input type="submit" value="Add Tutor" />
          </form>

          {loading && (
            <Space
              style={{
                width: "100vw",
                height: "100vh",
                position: "absolute",
                backgroundColor: "rgba(0,0,0,0.2)",
                top: "0",
                left: "0",
                display: "flex",
                justifyContent: "center",
                alignItems: "center",
              }}
            >
              <Spin size="large" />
            </Space>
          )}
        </Drawer>

        {/* Global loading spinner */}
        {load && (
          <Space
            style={{
              width: "100vw",
              height: "100vh",
              position: "absolute",
              backgroundColor: "rgba(0,0,0,0.2)",
              top: "0",
              left: "0",
              display: "flex",
              justifyContent: "center",
              alignItems: "center",
            }}
          >
            <Spin size="large" />
          </Space>
        )}
      </div>
    </Navbar>
  );
};

export default Tutor;
